/* ========= LCD_Helper.cpp ============== */
#include <Arduino.h>  // MUST be first
#include "LCD_Helper.h"
#include <math.h>
#include "temps.h"

// LCD_Helper.cpp (top of file, outside namespace)
static const uint8_t ICON_LOCK[8] = {
  0b01110, 0b10001,
  0b10001, 0b11111,
  0b11011, 0b11011,
  0b11111, 0b00000
};

static const uint8_t ICON_FROST[8] = {
  0b00100, 0b10101,
  0b01110, 0b10101,
  0b00100, 0b10101,
  0b01110, 0b00000
};

namespace LCD_

{
static bool iconsLoaded = false;

static void loadIcons(LiquidCrystal &lcd) {
  lcd.createChar(0, (uint8_t *)ICON_LOCK);
  lcd.createChar(1, (uint8_t *)ICON_FROST);
  iconsLoaded = true;
}

void renderTopRow(
  LiquidCrystal &lcd,
  MessageId msg,
  const char *const *messageTable,
  const DisplayModifiers &mods) {
  static uint8_t scrollIndex = 0;
  static unsigned long lastScroll = 0;
  const unsigned long SCROLL_MS = 300;
  constexpr unsigned long REFRESH_MS = 99;
  static unsigned long last_time = 0;

  static char shadow[17] = "                ";  // clear what LCD currently shows

  char text[24];
  char out[17];
  /*
  [ TEXT TEXT TEXT TEXT ][L][F]
   0..................13 14 15
  last the spaces reserved for icons Lock and Frost
*/
  if (!iconsLoaded) loadIcons(lcd);

  if (millis() - last_time < REFRESH_MS)
    return;

  last_time = millis();

  strncpy_P(text, (PGM_P)pgm_read_ptr(&messageTable[msg]), sizeof(text));
  text[sizeof(text) - 1] = '\0';

  /* ---------- Blink handling (FAULT) ---------- */
  if (mods.blink) {
    static bool visible = true;
    static unsigned long lastBlink = 0;

    if (millis() - lastBlink > 500) {
      lastBlink = millis();
      visible = !visible;
    }

    if (!visible) {
      // Desired output = blank line
      for (uint8_t i = 0; i < 16; i++)
        out[i] = ' ';
      out[16] = '\0';
    } else {
      goto BUILD_TEXT;
    }
  } else {
BUILD_TEXT:
    /* ---------- CAPS handling ---------- */
    if (mods.caps) {
      for (uint8_t i = 0; text[i]; i++) {
        if (text[i] >= 'a' && text[i] <= 'z')
          text[i] -= 32;
      }
    }

    /* ---------- Scroll handling ---------- */
    if (mods.scroll) {
      if (millis() - lastScroll > SCROLL_MS) {
        lastScroll = millis();
        scrollIndex++;
      }

      uint8_t len = strlen(text);
      if (scrollIndex >= len)
        scrollIndex = 0;

      for (uint8_t i = 0; i < 16; i++)
        out[i] = text[(scrollIndex + i) % len];
    } else {
      scrollIndex = 0;
      strncpy(out, text, 16);
    }

    out[16] = '\0';
  }

  /* ---------- DIFF-BASED WRITE ---------- */
  for (uint8_t col = 0; col < 16; col++) {
    if (out[col] != shadow[col]) {
      lcd.setCursor(col, 0);
      lcd.write(out[col]);
      shadow[col] = out[col];
    }
  }

  /* ---------- ICON RENDERING ---------- */
  static bool lockVisible = true;
  static unsigned long lastLockBlink = 0;

  if (mods.lockBlink) {
    if (millis() - lastLockBlink > 500) {
      lastLockBlink = millis();
      lockVisible = !lockVisible;
    }
  } else {
    lockVisible = true;
  }

  /* Column 14 = LOCK */
  char desiredLock =
    (mods.lock && lockVisible) ? 0 : ' ';

  /* Column 15 = FROST */
  char desiredFrost =
    mods.frost ? 1 : ' ';

  /* Diff-write icons */
  if (shadow[14] != desiredLock) {
    lcd.setCursor(14, 0);
    lcd.write(desiredLock);
    shadow[14] = desiredLock;
  }

  if (shadow[15] != desiredFrost) {
    lcd.setCursor(15, 0);
    lcd.write(desiredFrost);
    shadow[15] = desiredFrost;
  }
}

void renderBottomRow(
  LiquidCrystal &lcd,
  int16_t tin_x10,
  int16_t tcyl_x10,
  int16_t tstag_x10) {
  lcd.setCursor(0, 1);
  lcd.print("                ");  // remove any old text
  lcd.setCursor(0, 1);            // reposition cursor to start of the second line
  printTempPos_x10(lcd, tin_x10);
  printTempPos_x10(lcd, tcyl_x10);
  printTempSigned_x10(lcd, tstag_x10);
}


void printTempPos_x10(LiquidCrystal &lcd, int16_t t) {
  if (t < 0) {
    lcd.print(" --- ");  // this is out of range print dashes and return
    return;
  }

  int16_t whole = t / 10;
  int16_t frac = abs(t % 10);

  if (whole < 10)
    lcd.print(' '); // leading space if required
  lcd.print(whole);
  lcd.print('.');
  lcd.print(frac);
  lcd.print(' ');
}

void printTempSigned_x10(LiquidCrystal &lcd, int16_t t) {
  int16_t whole = abs(t) / 10;
  int16_t frac = abs(t % 10);

  // Explicit sign column
  if (t < 0)
    lcd.print('-');
  else
    lcd.print(' '); // leading space if required

  if (whole < 10)
    lcd.print(' ');
  lcd.print(whole);

  lcd.print('.');
  lcd.print(frac);
  lcd.print(' ');
}


}  // namespace LCD_
