#define VERSION "Version 6_5_3"
#define ARDUINO_AVR_NANO_EVERY
/*
SWHController_v6_5_3_STABLE`
  // HARDWARE REV: PCB v2.2.1  -- Modified for LCD connector and pin assignement - use FW v6_5_3 or higher
  // HARDWARE REV: PCB v2.1.1  -- includes a series 100 Ohm resistor for the ADC lines and 1k0 inline with digital switch
  // FIRMWARE BASELINE  v6_5_3 -- LCD Data pins reassigned for PCB layout - reqs HW 2.2.1 or higher
  // FIRMWARE BASELINE: v6_5_2 -- adds a safe ESD approach for unused pins
  // FIRMWARE BASELINE: v6_5_1 -- modifies the algorith to correct for the 100 Ohm resistor

### Digital I/O

* `PIN_PUMP` → relay / driver input
* `BUTTON_A` → pull-up, active LOW
* `LED_FROST` → output (currently unused)
*  LCD pins: `12, 11, 9, 8, 7, 6`

* ======================= LCD =======================
 * 4-bit HD44780 interface
 *
 * RS  -> D8
 * E   -> D9
 * D4  -> D6
 * D5  -> D5
 * D6  -> D4
 * D7  -> D3
 *
 * PCB: ControllerPCB Rev Jan2026v2
 *


### Electrical assumptions

* Button uses **INPUT_PULLUP**
* Pump relay logic level (**HIGH = ON to NFET)
* LCD is 4-bit parallel, not I²C
* Timing relies on `millis()` (no blocking)

*/

/* ======== SWHController.ino ========= */
#include <Arduino.h>
#include "pins.h"
#include <LiquidCrystal.h>
#include <math.h>
#include "button.h"
#include "LCD_Helper.h"
#include "SystemMode.h"
#include "temps.h"
#include "pump_control.h"

/* ---------- Forward declaration ---------- */
void clearPumpFault(PumpState &p, unsigned long now);

/* ======================= LCD ======================= */

// LCD pin mapping per PCB rev Jan2026v2
// lcd(rs, enable, d4, d5, d6, d7)

// RS = D8, E = D9
// D4 = D6, D5 = D5, D6 = D4, D7 = D3
// LiquidCrystal lcd(8, 9, 6, 5, 4, 3); assigned in pins.h

LiquidCrystal lcd(
  PIN_LCD_RS,
  PIN_LCD_EN,
  PIN_LCD_D4,
  PIN_LCD_D5,
  PIN_LCD_D6,
  PIN_LCD_D7
);

/* ================== Message Strings ================= */

const char msgStarting[] PROGMEM = "Starting    ";
const char msgAuto[] PROGMEM = "Auto          ";
const char msgOverride[] PROGMEM = "Override      ";
const char msgBackup[] PROGMEM = "Backup      ";

const char *const MessageTable[MSG_COUNT] PROGMEM = {
  msgStarting,
  msgAuto,
  msgOverride,
  msgBackup
};

button::State btnState;

MessageId messageForMode(SystemMode mode) {
  switch (mode) {
    case MODE_AUTO:
      return MSG_AUTO;
    case MODE_MANUAL:
      return MSG_OVERRIDE;
    case MODE_BACKUP:
      return MSG_BACKUP;
    default:
      return MSG_STARTING;
  }
}

/* =============  Setup ================== */
void setup() {
  pinMode(BUTTON_A, INPUT_PULLUP);
  pinMode(PIN_PUMP, OUTPUT);
  pinMode(LED_FROST, OUTPUT);
  digitalWrite(LED_FROST, LOW);  // not used in this version

  lcd.begin(16, 2);
  lcd.clear();
  lcd.print(F(VERSION));

  temps::begin();  // Prefill the array of temperatures
  delay(1500);
  lcd.clear();

  Serial.begin(115200);
  while (!Serial) { ; }  // safe on USB boards, harmless on UNO
  Serial.println(F("SWHController debug start"));
  initUnusedPins();
}

/*=============== Main Loop =============*/
void loop() {
  static SystemMode mode = MODE_AUTO;  // Initial state of the controller
  static PumpState pump;               // Pump controller state
  static bool faultLock = false;       // Latched until user clears it

  LCD_::DisplayModifiers mods{};
  bool requestOn = false;
  const unsigned long now = millis();  // this is the time for this loop iteration
  temps::update();

  if (temps::hasChanged()) {
    const auto &t = temps::avg();
    LCD_::renderBottomRow(
      lcd,
      t.tin_x10,
      t.tcyl_x10,
      t.tstag_x10);
  }

  /* ---------- Read button ---------- */
  if (button::pressed(btnState, now)) {
    clearPumpFault(pump, now);
    faultLock = false;

    switch (btnState) {
      case button::SHORT:
        mode = MODE_MANUAL;
        break;

      case button::LONG:
        mode = MODE_BACKUP;
        break;

      case button::TAP:
      default:
        mode = MODE_AUTO;
        break;
    }
  }

  /* ---------- Decide request ---------- */
  if (!faultLock) {
    switch (mode) {
      case MODE_MANUAL:
        requestOn = true;
        break;

      case MODE_AUTO:
        requestOn = temps::shouldPumpRun();
        break;

      default:
        requestOn = false;
        break;
    }
  }

  /* ---------- Pump control ---------- */
  bool pumpOn = updatePump(pump, requestOn, now);
  digitalWrite(PIN_PUMP, pumpOn);

  if (pump.maxRunFault) {
    faultLock = true;
    mode = MODE_AUTO;  // force auto
  }

  /* ---------- Display modifiers ---------- */
  mods.scroll = digitalRead(PIN_PUMP);
  mods.blink = !temps::sensor_inRange();
  mods.caps = (digitalRead(BUTTON_A) == LOW);
  mods.frost = temps::frostProtectionRequired();
  mods.lock = pump.lock;  // hold lock indication
  mods.lockBlink = pump.maxRunFault;

  MessageId msg = messageForMode(mode);
  LCD_::renderTopRow(lcd, msg, MessageTable, mods);

  // ---------- Serial debug ----------
  static unsigned long lastPrint = 0;
  if (now - lastPrint >= 1000) {
    lastPrint = now;

    Serial.print(F("now="));
    Serial.print(now);

    Serial.print(F(" pump="));
    Serial.print(pump.running ? F("ON") : F("OFF"));

    Serial.print(F(" req="));
    Serial.print(requestOn ? F("ON") : F("OFF"));

    Serial.print(F(" lock="));
    Serial.print(pump.lock ? F("HOLD") : F(" Go "));

    Serial.print(F("  "));
    Serial.print(pump.maxRunFault ? F("Overrun") : F(" -- "));

    Serial.println();
  }
}

/* ---------- Clear pump fault ---------- */

void clearPumpFault(PumpState &p, unsigned long /*now*/) {
  p.maxRunFault = false;
  p.lock = false;
  // DO NOT touch lastChange or onStart here
}

// Unused MCU pins are driven OUTPUT LOW to minimise ESD susceptibility
void initUnusedPins() {
  const uint8_t unusedDigitalPins[] = UNUSED_DIGITAL_PINS;
  for (uint8_t pin : unusedDigitalPins) {
    pinMode(pin, OUTPUT);
    digitalWrite(pin, LOW);
  }

  const uint8_t unusedAnalogPins[] = UNUSED_ANALOG_PINS;
  for (uint8_t pin : unusedAnalogPins) {
    pinMode(pin, OUTPUT);
    digitalWrite(pin, LOW);
  }
}
