/* ============= temps.cpp ============ */
#include <Arduino.h>
#include <math.h>
#include "temps.h"

namespace temps {

/* =========================
   Configuration
   ========================= */

constexpr uint8_t PIN_TOUT = A1;
constexpr uint8_t PIN_TIN = A2;
constexpr uint8_t PIN_TCYL = A3;
constexpr uint8_t PIN_TSTAG = A4;

constexpr uint8_t NUM_READINGS = 40;
constexpr uint32_t SAMPLE_INTERVAL_MS = 200;  // 40 × 200 ms = 8 s

// Pump thresholds (×10 °C)
constexpr int16_t CYL_DIFF_ON_x10 = 25;   // 2.5 °C
constexpr int16_t CYL_DIFF_OFF_x10 = 15;  // 1.5 °C
constexpr int16_t FROST_TEMP_x10 = 40;    // 4.0 °C

constexpr int16_t maxTemp = 1200;
constexpr int16_t minTemp = -300;

/* =========================
   ADC ring buffers
   ========================= */

static uint16_t tinBuf[NUM_READINGS];
static uint16_t toutBuf[NUM_READINGS];
static uint16_t tcylBuf[NUM_READINGS];
static uint16_t tstagBuf[NUM_READINGS];

static uint32_t tinSum, toutSum, tcylSum, tstagSum;
static uint8_t index;
static uint32_t lastSampleMs;

static Averages current{};
static Averages previous{};

/* =========================
   Thermistor conversion
   ========================= */

static double calcPTC_2K(uint16_t adc) {
  constexpr double R_PULLUP = 4700.0;  // 4k7
  constexpr double R_SERIES = 100.0;   // series resistor

  if (adc <= 1)
    return -273.15;
  if (adc >= 1022)
    return 150.0;

  // Correct divider math including series resistor
  double rTherm =
    (R_PULLUP * adc / (1023.0 - adc)) - R_SERIES;

  // Clamp to sane minimum
  if (rTherm < 100.0)
    rTherm = 100.0;

  // Empirical fit (your sensor)
  constexpr double A = 203.58;
  constexpr double B = -1529.9;

  return A * ::log(rTherm) + B;
}

static int16_t adcToTemp_x10(uint32_t adcAvg) {
  double temp = calcPTC_2K((uint16_t)adcAvg);
  return (int16_t)(temp * 10.0 + (temp >= 0 ? 0.5 : -0.5));
}

/* =========================
   Public API
   ========================= */

void begin() {
  constexpr uint16_t ADC_25C = 316;

  for (uint8_t i = 0; i < NUM_READINGS; i++) {
    tinBuf[i]  = ADC_25C;
    toutBuf[i] = ADC_25C;
    tcylBuf[i] = ADC_25C;
    tstagBuf[i] = ADC_25C;

    tinSum  += ADC_25C;
    toutSum += ADC_25C;
    tcylSum += ADC_25C;
    tstagSum += ADC_25C;
  }

  current = { 250, 250, 250, 250 };
  previous = current;
}

void update() {
  uint32_t now = millis();
  if (now - lastSampleMs < SAMPLE_INTERVAL_MS)
    return;

  lastSampleMs = now;

  // Remove old sample
  tinSum  -= tinBuf[index];
  toutSum -= toutBuf[index];
  tcylSum -= tcylBuf[index];
  tstagSum -= tstagBuf[index];

  // Read ADCs
  tinBuf[index]  = analogRead(PIN_TIN);
  toutBuf[index] = analogRead(PIN_TOUT);
  tcylBuf[index] = analogRead(PIN_TCYL);
  tstagBuf[index] = analogRead(PIN_TSTAG);

  // Add new
  tinSum  += tinBuf[index];
  toutSum += toutBuf[index];
  tcylSum += tcylBuf[index];
  tstagSum += tstagBuf[index];

  index = (index + 1) % NUM_READINGS;

  // Convert averaged ADC → temperature
  current.tin_x10   = adcToTemp_x10(tinSum / NUM_READINGS);
  current.tout_x10  = adcToTemp_x10(toutSum / NUM_READINGS);
  current.tcyl_x10  = adcToTemp_x10(tcylSum / NUM_READINGS);
  current.tstag_x10 = adcToTemp_x10(tstagSum / NUM_READINGS);
}

const Averages &avg() {
  return current;
}

/* =========================
   Pump logic
   ========================= */

bool shouldPumpRun() {
  return (current.tstag_x10 - current.tin_x10) >= CYL_DIFF_ON_x10;
}

bool frostProtectionRequired() {
  return current.tstag_x10 <= FROST_TEMP_x10;
}

bool hasChanged() {
  bool changed =
    current.tin_x10 != previous.tin_x10 
    || current.tout_x10  != previous.tout_x10 
    || current.tcyl_x10  != previous.tcyl_x10 
    || current.tstag_x10 != previous.tstag_x10;

  if (changed)
    previous = current;

  return changed;
}

bool sensor_inRange() {
  return current.tin_x10 >= minTemp && current.tin_x10 
  <= maxTemp && current.tout_x10  >= minTemp && current.tout_x10 
  <= maxTemp && current.tcyl_x10  >= minTemp && current.tcyl_x10 
  <= maxTemp && current.tstag_x10 >= minTemp && current.tstag_x10 
  <= maxTemp;
}

}  // namespace temps
