/* ======== SWHController.ino ========= */
#include <Arduino.h>
#include <LiquidCrystal.h>
#include <math.h>

#include "button.h"
#include "LCD_Helper.h"
#include "SystemMode.h"
#include "temps.h"
#include "button_config.h"
#include "pump_control.h"

/* ---------- Forward declaration ---------- */
void clearPumpFault(PumpState &p, unsigned long now);

/* ======================= LCD ======================= */
//         (rs, enable, d4, d5, d6, d7)
LiquidCrystal lcd(12, 11, 9, 8, 7, 6); // revised layout

/* ================== Message Strings ================= */

const char msgStarting[] PROGMEM = "Starting    ";
const char msgAuto[] PROGMEM     = "Auto          ";
const char msgOverride[] PROGMEM = "Override      ";
const char msgBackup[] PROGMEM   = "Backup      ";

const char *const MessageTable[MSG_COUNT] PROGMEM =
{
    msgStarting,
    msgAuto,
    msgOverride,
    msgBackup
};

button::State btnState;

MessageId messageForMode(SystemMode mode)
{
    switch (mode)
    {
    case MODE_AUTO:
        return MSG_AUTO;
    case MODE_MANUAL:
        return MSG_OVERRIDE;
    case MODE_BACKUP:
        return MSG_BACKUP;
    default:
        return MSG_STARTING;
    }
}

/* =============  Setup ================== */
void setup()
{
    pinMode(BUTTON_A, INPUT_PULLUP);
    pinMode(PIN_PUMP, OUTPUT);
    pinMode(LED_FROST, OUTPUT);
    digitalWrite(LED_FROST, LOW); // not used in this version

    lcd.begin(16, 2);
    lcd.clear();
    lcd.print(F("Control v6_5.0"));

    temps::begin();   // Prefill the array of temperatures
    delay(1500);
    lcd.clear();

    Serial.begin(115200);
while (!Serial) { ; }   // safe on USB boards, harmless on UNO
Serial.println(F("SWHController debug start"));

}

/*=============== Main Loop =============*/
void loop()
{
    static SystemMode mode = MODE_AUTO;   // Initial state of the controller
    static PumpState pump;                // Pump controller state
    static bool faultLock = false;        // Latched until user clears it

    LCD_::DisplayModifiers mods{};
    bool requestOn = false;
    const unsigned long now = millis(); // this is the time for this loop iteration
    temps::update();

    if (temps::hasChanged())
    {
        const auto &t = temps::avg();
        LCD_::renderBottomRow(
            lcd,
            t.tin_x10,
            t.tcyl_x10,
            t.tstag_x10
        );
    }

    /* ---------- Read button ---------- */
    if (button::pressed(btnState, now))
    {
        clearPumpFault(pump, now);
        faultLock = false;

        switch (btnState)
        {
        case button::SHORT:
            mode = MODE_MANUAL;
            break;

        case button::LONG:
            mode = MODE_BACKUP;
            break;

        case button::TAP:
        default:
            mode = MODE_AUTO;
            break;
        }
    }

    /* ---------- Decide request ---------- */
    if (!faultLock)
    {
        switch (mode)
        {
        case MODE_MANUAL:
            requestOn = true;
            break;

        case MODE_AUTO:
            requestOn = temps::shouldPumpRun();
            break;

        default:
            requestOn = false;
            break;
        }
    }

    /* ---------- Pump control ---------- */
    bool pumpOn = updatePump(pump, requestOn, now);
    digitalWrite(PIN_PUMP, pumpOn);

    if (pump.maxRunFault)
    {
        faultLock = true;
        mode = MODE_AUTO; // force auto
    }

    /* ---------- Display modifiers ---------- */
    mods.scroll     = digitalRead(PIN_PUMP);
    mods.blink      = !temps::sensor_inRange();
    mods.caps       = (digitalRead(BUTTON_A) == LOW);
    mods.frost      = temps::frostProtectionRequired();
    mods.lock       = pump.lock;       // hold lock indication
    mods.lockBlink  = pump.maxRunFault;

    MessageId msg = messageForMode(mode);
    LCD_::renderTopRow(lcd, msg, MessageTable, mods);

    // ---------- Serial debug ----------
static unsigned long lastPrint = 0;
if (now - lastPrint >= 1000) {
    lastPrint = now;

    Serial.print(F("now="));
    Serial.print(now);

    Serial.print(F(" pump="));
    Serial.print(pump.running ? F("ON") : F("OFF"));

    Serial.print(F(" req="));
    Serial.print(requestOn ? F("ON") : F("OFF"));

    Serial.print(F(" lock="));
    Serial.print(pump.lock ? F("HOLD") : F(" Go "));

    Serial.print(F("  "));
    Serial.print(pump.maxRunFault ? F("Overrun") : F(" -- "));

    Serial.println();
}

}

/* ---------- Clear pump fault ---------- */

void clearPumpFault(PumpState &p, unsigned long /*now*/)
{
    p.maxRunFault = false;
    p.lock = false;
    // DO NOT touch lastChange or onStart here
}

